//---------------------------------------------------------------------------
/*

  The goal of this example is to show how to use two different
  drivers in the same application.  Typical case is to have one
  board for Analog I/O and a second board for Digital I/O (or
  counter timer functions).

  This example performs a polled mode AI sample from one driver(board)
  and a polled mode DI sample from another driver(board).

  When using two different drivers in an application, it is necessary
  to make use of the SelectDriverLINX function to load the required
  mini-port driver for the specific hardware.  The higher level functions
  of DriverLINX are hardware independent and work in concert with a mini-port
  driver for each specific hardware family.  The SelectDriverLNX function
  takes the instance handle as a parameter.  The instance handle is generated
  when the driver is opened (OpenDriverLINX).



*/
#include <vcl.h>
#pragma hdrstop

#include "MultiSRmain.h"
#include "drvlinx.h"   /* DriverLINX API */
#include "dlcodes.h"   /* DriverLINX error codes and macros */
// be sure to add c:\drvlinx4\dlapi\drvlnxbc.lib to the project

// DriverLINX variables and function prototypes

DL_SERVICEREQUEST AISR;    // declare the Service Request Structure
DL_SERVICEREQUEST DISR;
HANDLE hInstanceAI;       // instance handle for the driver
HANDLE hInstanceDI;
HANDLE hWin;           // handle to this application
int AI_DeviceNum = 0;   // set device number of your AI board here
int DI_DeviceNum = 0;   // set device number of your DI board here
// is is OK to have a device 0 for each unique driver

// variables used for error handling
char errString[100];
DWORD size=50;
int errNumber;
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"
TForm1 *Form1;
//---------------------------------------------------------------------------
__fastcall TForm1::TForm1(TComponent* Owner)
        : TForm(Owner)
{
}
//---------------------------------------------------------------------------
void __fastcall TForm1::cmdOpenClick(TObject *Sender)
{
const char driverName1[20] = "kpci3100";  // analog input
const char driverName2[20] = "kpcipio";   // digital input
// change the driver names above to match your installed hardware

       // get the handle of this application's window
      hWin = Form1->Handle;

	// call OpenDriverLINX()
	hInstanceAI = (HANDLE)OpenDriverLINX( hWin, driverName1 );
        // use name of your driver to avoid the OpenDriverLINX Dialog box.
        //
        // open the other driver too
        hInstanceDI = (HANDLE)OpenDriverLINX( hWin, driverName2 );

        Form1->SetFocus();  // set focus back to our application
        // allow Initialization to take place now....
        cmdInit->Enabled = True;
}
//---------------------------------------------------------------------------

void __fastcall TForm1::cmdInitClick(TObject *Sender)
{
     	memset( &AISR, 0, sizeof( DL_SERVICEREQUEST ) );
	DL_SetServiceRequestSize( AISR );

	AISR.hWnd = hWin;
        AISR.operation = INITIALIZE;  // carry out an initialization operation
	AISR.device = AI_DeviceNum;  // the device number assigned in DLinx Config Panel
	AISR.subsystem = DEVICE;
	AISR.mode = OTHER;
    	// call DriverLINX() to execute the task specified by the
        // members of the SR structure
	if( DriverLINX( &AISR ) != NoErr )
         {// respond to error
           errNumber = getErrCode(AISR.result);
        // errNumber corresponds to DriverLINX documentation
        ReturnMessageString(AISR.hWnd,AISR.result,errString,size);
        // errString corresponds to DriverLINX documentation
        lblAIStatus->Caption = errString;
        // don't allow AI sampling....
        cmdReadAI->Enabled =True;
          }
        else
        {// all is well, allow AI sampling...
        cmdReadAI->Enabled =True;
        }

    // do the same for the second Service Request, DISR
       	memset( &DISR, 0, sizeof( DL_SERVICEREQUEST ) );
	DL_SetServiceRequestSize( DISR );

	DISR.hWnd = hWin;
        DISR.operation = INITIALIZE;  // carry out an initialization operation
	DISR.device = DI_DeviceNum;  // the device number assigned in DLinx Config Panel
	DISR.subsystem = DEVICE;
	DISR.mode = OTHER;
       	if( DriverLINX( &DISR ) != NoErr )
         {// respond to error
           errNumber = getErrCode(DISR.result);
        // errNumber corresponds to DriverLINX documentation
        ReturnMessageString(DISR.hWnd,DISR.result,errString,size);
        // errString corresponds to DriverLINX documentation
        lblDIStatus->Caption = errString;
        // don't allow DI sampling....
        cmdReadDI->Enabled =True;
          }
        else
        {// all is well, allow DI sampling...
        cmdReadDI->Enabled =True;
        }

}
//---------------------------------------------------------------------------


void __fastcall TForm1::cmdReadDIClick(TObject *Sender)
{
        SelectDriverLINX(hInstanceDI); // use instance for DIO board driver
	DISR.hWnd = hWin;
	DISR.device = DI_DeviceNum;
	DISR.subsystem = DI; // use the Digital Input subsystem of the board
	DISR.mode = POLLED;
	DISR.operation = START;
 	DISR.channels.nChannels = 1;  // a single channel will be sampled
	DISR.channels.chanGain[ 0 ].channel = 0;  // it is channel 0
        DISR.taskFlags = NO_SERVICESTART | NO_SERVICEDONE;

	// call DriverLINX()
	if( DriverLINX( &DISR ) != NoErr )
        {
	errNumber = getErrCode(DISR.result);
        // errNumber corresponds to DriverLINX documentation
        ReturnMessageString(DISR.hWnd,DISR.result,errString,size);
        // errString corresponds to DriverLINX documentation
        lblDIStatus->Caption = errString;
        lblDIValue->Caption = "";
	}
        else
        {
         lblDIValue->Caption = DISR.status.u.ioValue;
         lblDIStatus->Caption = "No Error";
        }
}
//---------------------------------------------------------------------------

void __fastcall TForm1::cmdReadAIClick(TObject *Sender)
{
        SelectDriverLINX(hInstanceAI);  // use instance for Analog board driver
	AISR.hWnd = hWin;
	AISR.device = AI_DeviceNum;
	AISR.subsystem = AI; // use the Analog Input subsystem of the board
	AISR.mode = POLLED;
	AISR.operation = START;
 	AISR.channels.nChannels = 1;  // a single channel
	AISR.channels.chanGain[ 0 ].channel = 0;  // it is channel 0
        AISR.channels.chanGain[ 0 ].gainOrRange = Gain2Code(0,AI,-1.0);
        AISR.taskFlags = NO_SERVICESTART | NO_SERVICEDONE;

	// call DriverLINX()
	if( DriverLINX( &AISR ) != NoErr )
        {
         errNumber = getErrCode(AISR.result);
        // errNumber corresponds to DriverLINX documentation
        ReturnMessageString(AISR.hWnd,AISR.result,errString,size);
        // errString corresponds to DriverLINX documentation
        lblAIStatus->Caption = errString;
        lblAIValue->Caption = "";
	}
        else
        {
        float volts;
        int result;
        result=Code2Volts(0,AI,AISR.status.u.ioValue,&volts); //Convert to volts
        char str[25];
        gcvt(volts, 4, str); // format volts
        lblAIValue->Caption = str;
        lblAIStatus->Caption = "No Error";
        }
}
//---------------------------------------------------------------------------


void __fastcall TForm1::cleanUpDriverLINX(TObject *Sender)
{
  // close the drivers
        SelectDriverLINX(hInstanceAI);
  	CloseDriverLINX(hInstanceAI); //close the DriverLINX driver
	hInstanceAI=NULL; //for safety, make sure hInstance isn't pointing to anything
	delete(&AISR); //de-allocate the memory used by the service request

         // don't close same driver a second time
        // will raise an error
        // same thing for other driver
        SelectDriverLINX(hInstanceDI);
     	CloseDriverLINX(hInstanceDI); //close the DriverLINX driver
	hInstanceDI=NULL; //for safety, make sure hInstance isn't pointing to anything
	delete(&DISR);
}
//---------------------------------------------------------------------------

